/*******************************************************************************
 * Licensed Materials - Property of IBM
 * © Copyright IBM Corporation 2014, 2019. All Rights Reserved.
 * 
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp. 
 *******************************************************************************/
// NOTE: THIS IS A GENERATED FILE. DO NOT EDIT DIRECTLY!
dojo.provide("com.ibm.jdojox.util.OrderedMap");

dojo.require("dojo.string");

(function() {
var string= dojo.string;

/**
 * Class is used to store items which can be retrieved by their identifier and
 * can be retrieved in a specific order
 * 
 * @param <T> The item type which is being stored
 */
var OrderedMap= dojo.declare("com.ibm.jdojox.util.OrderedMap", null, {
	"-chains-": { constructor: "manual" },

	/**
	 * A list of all elements sorted by Index
	 */
	//elementByIndex: null,

	/**
	 * A list of all identifiers sorted by Index
	 */
	//identifierByIndex: null,

	/**
	 * A list of all elements sorted by identifier
	 */
	//elementByIdentifier: null,

	/**
	 * A list of all indexes sorted by identifier
	 */
	//indexByIdentifier: null,

	/**
	 * Determines the first index which is incorrect
	 */
	reindex: 0,

	/**
	 * Creates an empty OrderedMap
	 */
	constructor: function() {
		this.elementByIndex= [];
		this.identifierByIndex= [];
		this.elementByIdentifier= {};
		this.indexByIdentifier= {};
		this.reindex= -1;
	},

	/**
	 * Gets an element by an identifier.
	 * 
	 * @param identifier The identifier to look up the element by
	 * @return The element which the id is located at, Undefined.VALUE() if no
	 *         such element is found
	 */
	getElementByIdentifier: function(identifier) {
		return this.elementByIdentifier[identifier];
	},

	/**
	 * Gets an element by an index
	 * 
	 * @param index The index to look up the element by
	 * @return The element which the index is located at, Undefined.VALUE() if
	 *         no such element is found
	 */
	getElementByIndex: function(index) {
		if (index < this.identifierByIndex.length) {
			return this.elementByIndex[index];
		} else {
			return undefined;
		}
	},

	/**
	 * Gets an identifier by an index
	 * 
	 * @param index The index to look up the id by
	 * @return The identifier which the index is located at, Undefined.VALUE()
	 *         if no such element is found
	 */
	getIdentifierByIndex: function(index) {
		if (index < this.identifierByIndex.length) {
			return this.identifierByIndex[index];
		} else {
			return undefined;
		}
	},

	/**
	 * Gets an index by an identifier.
	 * 
	 * <b>Note:</b> Could be slow, it may require reindexing.
	 * 
	 * @param identifier
	 *            The identifier to look up the index by
	 * @return The index which the id is located at, -1 if no such element is
	 *         found
	 */
	getIndexByIdentifier: function(identifier) {
		var index= this.indexByIdentifier[identifier];
		if (index == null) {
			return -1;
		} else if (this.reindex === -1 || index < this.reindex) {
			return index;
		} else {
			this._reindexIndexByIdentifier(this.reindex);
			return this.indexByIdentifier[identifier];
		}
	},

	/**
	 * Reindex all elements after a specific element
	 * 
	 * @param start The element which is the first that requires reindexing
	 */
	_reindexIndexByIdentifier: function(start) {
		for (var i= start; i < this.getSize(); i++){
			this.indexByIdentifier[this.identifierByIndex[i]]= i;
		}
	},

	/**
	 * Flushes the OrderedMap of all values
	 */
	clear: function() {
		this.elementByIndex= [];
		this.identifierByIndex= [];
		this.elementByIdentifier= {};
		this.indexByIdentifier= {};
		this.reindex= -1;
	},

	/**
	 * Gets an array of all identifiers sorted by their index
	 * 
	 * @return An array of all identifiers sorted by their index
	 */
	getKeyArray: function() {
		return this.identifierByIndex;
	},

	/**
	 * Gets an array of all elements sorted by their index
	 * 
	 * @return An array of all elements sorted by their index
	 */
	getValueArray: function() {
		return this.elementByIndex;
	},

	iterator: function() {
		return new OrderedMapIterator(this);
	},

	/**
	 * Gets the number of elements in the OrderedMap
	 * 
	 * @return The number of elements in the OrderedMap
	 */
	getSize: function() {
		return this.elementByIndex.length;
	},

	/**
	 * Determines if the element with the specific identifier exists
	 * 
	 * @param identifier
	 *            The identifier to look up the element by
	 * @return True if the element exists, false otherwise
	 */
	hasElementWithIdentifier: function(identifier) {
		return (typeof (this.indexByIdentifier[identifier]) !== 'undefined');
	},

	/**
	 * Adds an element at a specific index.
	 * 
	 * <b>Note:</b> Unless element is added to the end, will require reindexing
	 * to correct getIndexByIdentifier(...).
	 * 
	 * @param identifier
	 *            The identifier which is used to look up the element
	 * @param element
	 *            The element to be added
	 * @param index
	 *            The index which to add the element at
	 * @return true on success, false on error
	 */
	addElementAtIndex: function(identifier, element, index) {
		if (this.getElementByIdentifier(identifier) != null) {
			console.error(string.substitute(OrderedMap.OrderedMap_ERROR_IDENTIFIER_CONFLICT, [identifier ]));
		} else if (index < 0 || this.getSize() < index) {
			console.error(string.substitute(OrderedMap.OrderedMap_ERROR_INDEX_OUT_OF_RANGE, [index , this.getSize()]));
		} else if (this.getSize() === index) {
			return this.addElementAtEnd(identifier, element);
		} else {
			if (this.reindex === -1 || index < this.reindex) {
				this.reindex= index;
			}
			this.elementByIdentifier[identifier]= element;
			this.indexByIdentifier[identifier]= index;
			this.elementByIndex.splice(index, 0, element);
			this.identifierByIndex.splice(index, 0, identifier);
			return true;
		}
		return false;
	},

	/**
	 * Adds an element at the end of the array
	 * 
	 * @param identifier The identifier which is used to look up the element
	 * @param element The element to be added
	 * @return true on success, false on error
	 */
	addElementAtEnd: function(identifier, element) {
		if (this.getElementByIdentifier(identifier) != null) {
			console.error(string.substitute(OrderedMap.OrderedMap_ERROR_IDENTIFIER_CONFLICT, [identifier != null ? identifier : "null" ]));
		} else {
			this.elementByIdentifier[identifier]= element;
			this.indexByIdentifier[identifier]= this.elementByIndex.length;
			this.elementByIndex.push(element);
			this.identifierByIndex.push(identifier);
			return true;
		}
		return false;
	},

	/**
	 * Removes an element.
	 * 
	 * <b>Note:</b> Unless last element is removed, will require reindexing to
	 * correct getIndexByIdentifier(...).
	 * 
	 * @param identifier
	 *            The identifier which to remove
	 * @param index
	 *            The index which to remove
	 * @return The removed element, Undefined.VALUE() on error
	 */
	_remove: function(identifier, index) {
		if (this.getSize() <= index || index < 0) {
			console.error(string.substitute(OrderedMap.OrderedMap_ERROR_INDEX_OUT_OF_RANGE, [index , this.getSize()]));
		} else if (this.getElementByIdentifier(identifier) == null) {
			console.error(string.substitute(OrderedMap.OrderedMap_ERROR_IDENTIFIER_CONFLICT, [identifier ]));
		} else {
			if (this.reindex === -1 || index < this.reindex) {
				this.reindex= index;
			}
			delete this.elementByIdentifier[identifier];
			delete this.indexByIdentifier[identifier];
			this.identifierByIndex.splice(index, 1);
			var removedElement= this.elementByIndex.splice(index, 1)[0];
			if (this.getSize() <= this.reindex) {
				this.reindex= -1;
			}
			return removedElement;
		}
		return undefined;
	},

	/**
	 * Removes an element by its index.
	 * 
	 * <b>Note:</b> Unless last element is removed, will require reindexing to
	 * correct getIndexByIdentifier(...).
	 * 
	 * @param index
	 *            The index which to remove an element by
	 * @return The removed element, Undefined.VALUE() if element does not exist
	 */
	removeElementByIndex: function(index) {
		var id= this.getIdentifierByIndex(index);
		return this._remove(id, index);
	},

	/**
	 * Removes an element by its identifier.
	 * 
	 * <b>Note:</b> Could be slow, it may require reindexing. <b>Note:</b>
	 * Unless last element is removed, will require reindexing to correct
	 * getIndexByIdentifier(...).
	 * 
	 * @param identifier
	 *            The identifier which to remove an element by
	 * @return The removed element, Undefined.VALUE() if element does not exist
	 */
	removeElementByIdentifier: function(identifier) {
		var index= this.getIndexByIdentifier(identifier);
		if (index < 0) {
			return undefined;
		}
		return this._remove(identifier, index);
	},

	$interfaces: {
		'com.ibm.jdojox.util.IIterable': true
	}
});
/**
 * The iterator for OrderedMap
 *
 * @param <T> The item type which is being stored
 */
var OrderedMapIterator= dojo.declare("com.ibm.jdojox.util.OrderedMap.OrderedMapIterator", null, {
	"-chains-": { constructor: "manual" },

	/**
	 * The map being iterated across
	 */
	//map: null,

	/**
	 * The current position of the iterator
	 */
	position: 0,

	/**
	 * A basic constructor
	 * 
	 * @param map The map being iterated across
	 */
	constructor: function(map) {
		this.map= map;
		this.position= -1;
	},

	hasNext: function() {
		return ((this.position + 1) < this.map.elementByIndex.length);
	},

	next: function() {
		this.position++;
		if (this.position < this.map.elementByIndex.length) {
			return {key: this.map.identifierByIndex[this.position], value: this.map.elementByIndex[this.position]};
		}
		return null;
	},

	remove: function() {
		if (0 < this.position && this.position < this.map.elementByIndex.length) {
			this.map.removeElementByIndex(this.position--);
		}
	},

	$interfaces: {
		'com.ibm.jdojox.util.IIterator': true
	}
});

OrderedMap.OrderedMap_ERROR_IDENTIFIER_CONFLICT= "Identifier <${0}> conflicts with existing element.";
OrderedMap.OrderedMap_ERROR_INDEX_OUT_OF_RANGE= "Index <${0}> is out of range <${1}>.";

})();
