/**
 * IBM Confidential
 * OCO Source Materials
 * 5725-H03
 * (C) Copyright IBM Corp. 2018, 2021
 * The source code for this program is not published or 
 * otherwise divested of its trade secrets, irrespective of 
 * what has been deposited with the U.S. Copyright Office.
 */
package com.ibm.pdp.micropattern.analyzer;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;

import com.ibm.pdp.engine.IAnalyzerResult;
import com.ibm.pdp.engine.IMicroPatternProcessingContext;
import com.ibm.pdp.micropattern.analyzer.plugin.IMPAnalyzer;
import com.ibm.pdp.micropattern.analyzer.plugin.IMPExtendedAnalyzer;
import com.ibm.pdp.micropattern.analyzer.tool.MPDesignReference;
import com.ibm.pdp.util.Util;

public class MPAnalyzerManager {

    public final static String copyright = "Licensed Materials - Property of IBM\n5725-H03\n(C) Copyright IBM Corp. 2018, 2021.   All rights reserved.\nUS Government Users Restricted Rights - Use, duplication or disclosure restricted by GSA ADP Schedule Contract with IBM Corp.";

    // extension point definition
    private static final String _MP_ANALYZER_EXTENSION_POINT = "mpAnalyzer"; //$NON-NLS-1$
    private static final String _MP_EXTENDED_ANALYZER_EXTENSION_POINT = "mpExtendedAnalyzer"; //$NON-NLS-1$
    private static final String _DOT = "dot";

    // singleton
    private static MPAnalyzerManager _instance = null;

    // runtime attributes to manage extension point implementors
    private Map<String, IMPAnalyzer> _mpAnalyzerImplementors = new HashMap<String, IMPAnalyzer>();
    private Map<String, IMPExtendedAnalyzer> _mpExtendedAnalyzerImplementors = new HashMap<String, IMPExtendedAnalyzer>();

    private static MPAnalyzerManager getInstance() {
        if (_instance == null) {
            _instance = new MPAnalyzerManager();

            // load all extension point implementors
            _instance.loadExtensionPoints(_MP_ANALYZER_EXTENSION_POINT);
            _instance.loadExtensionPoints(_MP_EXTENDED_ANALYZER_EXTENSION_POINT);
        }

        return _instance;
    }

    /**
     * Return all 'mpAnalyzer' extension point implementors.
     */
    public static Map<String, IMPAnalyzer> getMPAnalyzerImplementors() {
        return getInstance()._mpAnalyzerImplementors;
    }

    /**
     * Return all 'mpExtendedAnalyzer' extension point implementors.
     */
    public static Map<String, IMPExtendedAnalyzer> getMPExtendedAnalyzerImplementors() {
        return getInstance()._mpExtendedAnalyzerImplementors;
    }

    /**
     * Search and load all extension points.
     */
    private void loadExtensionPoints(String extensionPoint) {
        IExtensionRegistry extRegistry = Platform.getExtensionRegistry();
        IExtensionPoint extPoint = extRegistry.getExtensionPoint("com.ibm.pdp.micropattern.analyzer", extensionPoint); //$NON-NLS-1$
        if (extPoint != null) {
            IExtension[] extensions = extPoint.getExtensions();
            for (int i = 0; i < extensions.length; i++) {
                IConfigurationElement[] configElements = extensions[i].getConfigurationElements();
                for (int j = 0; j < configElements.length; j++) {
                    IConfigurationElement configElement = configElements[j];
                    if (configElement.getName().equals(extensionPoint)) {
                        // process the configuration element
                        processConfigElement(configElement);
                    }
                }
            }
        }
    }

    private void processConfigElement(IConfigurationElement element) {
        try {
            if (element.getName().equals(_MP_ANALYZER_EXTENSION_POINT)) {
                IMPAnalyzer mpAnalyzer = (IMPAnalyzer) element.createExecutableExtension("class"); //$NON-NLS-1$

                // initialize the micro-pattern instance for the extension-point 
                mpAnalyzer.setNamespace(element.getAttribute("namespace")); //$NON-NLS-1$
                mpAnalyzer.setIdentifier(element.getAttribute("identifier")); //$NON-NLS-1$
                mpAnalyzer.setDescription(element.getAttribute("description")); //$NON-NLS-1$
                mpAnalyzer.setArea(element.getAttribute("area")); //$NON-NLS-1$
                boolean contentAssist = false;
                String attContentAssist = element.getAttribute("contentAssist");
                if (attContentAssist != null && Boolean.toString(true).equals(attContentAssist))
                    contentAssist = true;
                mpAnalyzer.setContentAssist(contentAssist);

                // initialize the parameters for this micro-pattern
                for (IConfigurationElement parameterNode : element.getChildren("mpParameter")) {
                    boolean isOptional = false;
                    String attIsOptional = parameterNode.getAttribute("isOptional");
                    if (attIsOptional != null && Boolean.toString(true).equals(attIsOptional))
                        isOptional = true;
                    boolean isDefault = false;
                    String attIsDefault = parameterNode.getAttribute("isDefault");
                    if (attIsDefault != null && Boolean.toString(true).equals(attIsDefault))
                        isDefault = true;
                    MPParameter mpParameter = new MPParameter(parameterNode.getAttribute("name"), parameterNode.getAttribute("description"),
                            parameterNode.getAttribute("type"), parameterNode.getAttribute("designType"), isOptional, isDefault,
                            parameterNode.getAttribute("defValue"));

                    // initialize the list of values for this parameter
                    for (IConfigurationElement parameterValueNode : parameterNode.getChildren("mpParameterValue")) {
                        MPParameterValue mpParameterValue = new MPParameterValue(parameterValueNode.getAttribute("value"),
                                parameterValueNode.getAttribute("description"));
                        mpParameter.getParameterValues().add(mpParameterValue);
                    }

                    if (!mpAnalyzer.getKeyedParameters().containsKey(mpParameter.getName())) { // to avoid double in the list
                        mpAnalyzer.getParameters().add(mpParameter);
                        mpAnalyzer.getKeyedParameters().put(mpParameter.getName(), mpParameter);
                    }
                }

                // initialize the patterns for this micro-pattern
                for (IConfigurationElement patternNode : element.getChildren("mpPattern")) {
                    MPPattern mpPattern = new MPPattern(patternNode.getAttribute("patternId"), patternNode.getAttribute("description"));

                    // initialize the parameters for this pattern
                    for (IConfigurationElement parameterNode : patternNode.getChildren("mpParameter")) {
                        boolean isOptional = false;
                        String attIsOptional = parameterNode.getAttribute("isOptional");
                        if (attIsOptional != null && Boolean.toString(true).equals(attIsOptional))
                            isOptional = true;
                        boolean isDefault = false;
                        String attIsDefault = parameterNode.getAttribute("isDefault");
                        if (attIsDefault != null && Boolean.toString(true).equals(attIsDefault))
                            isDefault = true;
                        MPParameter mpParameter = new MPParameter(parameterNode.getAttribute("name"), parameterNode.getAttribute("description"),
                                parameterNode.getAttribute("type"), parameterNode.getAttribute("designType"), isOptional, isDefault,
                                parameterNode.getAttribute("defValue"));

                        // initialize the list of values for this parameter
                        for (IConfigurationElement parameterValueNode : parameterNode.getChildren("mpParameterValue")) {
                            MPParameterValue mpParameterValue = new MPParameterValue(parameterValueNode.getAttribute("value"),
                                    parameterValueNode.getAttribute("description"));
                            mpParameter.getParameterValues().add(mpParameterValue);
                        }

                        if (!mpAnalyzer.getKeyedParameters().containsKey(mpParameter.getName())) { // to avoid double in the list
                            mpPattern.getParameters().add(mpParameter);
                            mpPattern.getKeyedParameters().put(mpParameter.getName(), mpParameter);
                        }
                    }

                    mpAnalyzer.getKeyedPatterns().put(mpPattern.getPatternId(), mpPattern);
                }

                getMPAnalyzerImplementors().put(mpAnalyzer.getIdentifier(), mpAnalyzer);
            }
            else if (element.getName().equals(_MP_EXTENDED_ANALYZER_EXTENSION_POINT)) {
                IMPExtendedAnalyzer mpExtendedAnalyzer = (IMPExtendedAnalyzer) element.createExecutableExtension("class"); //$NON-NLS-1$

                // initialize the micro-pattern instance for the extension-point
                mpExtendedAnalyzer.setNamespace(element.getAttribute("namespace")); //$NON-NLS-1$
                mpExtendedAnalyzer.setIdentifier(element.getAttribute("identifier")); //$NON-NLS-1$

                getMPExtendedAnalyzerImplementors().put(mpExtendedAnalyzer.getIdentifier(), mpExtendedAnalyzer);
            }
        } catch (CoreException e) {
            //throw Util.rethrow(e);
        }
    }

    private static IMPAnalyzer findAnalyzer(String text) {
        // step1: standard analyzers are examined
        List<IMPAnalyzer> genericAnalyzers = new ArrayList<IMPAnalyzer>();
        for (IMPAnalyzer mpAnalyzer : getMPAnalyzerImplementors().values()) {
            if (!mpAnalyzer.getIdentifier().endsWith("*")) {
                if (mpAnalyzer.accept(text)) {
                    return mpAnalyzer;
                }
            }
            else {
                genericAnalyzers.add(mpAnalyzer);
            }
        }

        // step2: generic analyzers are examined 
        for (IMPAnalyzer mpAnalyzer : genericAnalyzers) {
            if (mpAnalyzer.accept(text)) {
                return mpAnalyzer;
            }
        }

        return null;
    }

    public static IAnalyzerResult analyze( String text, int beginIdx, IMicroPatternProcessingContext context )
    	throws MPException
    {
       	return analyze2( text, beginIdx, context );

        /* Old content for the method (work item 34588)
    	return analyze1( text, beginIdx, context );
    	*/
    }

    
    /**
     * Analyze the micro-pattern syntax, specified in the parameter 'text', at the index 'beginIdx'.
     * If the analyze is successful returns an instance of IAnalyzerResult.
     * Null otherwise.
     * The parameter 'context' describes the micro-pattern context. This context is required for the access to design files content,
     * via the extended analyser to check the micro-pattern parameters.
     * If this parameter is null, the extended analyzer is not activated.
     */
    public static IAnalyzerResult analyze1(String text, int beginIdx, IMicroPatternProcessingContext context)
    		throws MPException {
        IMPAnalyzer mpAnalyzer = null;
        IAnalyzerResult mpAnalyzerResult = null;
        StringBuilder headerBuffer = new StringBuilder("");
        StringBuilder parameterBuffer = new StringBuilder("");
        Map<String, String> allPairs = new LinkedHashMap<String, String>();
        String lineSeparator = Util.determineDelimiterOfV2(text);

        InputStream inputStream = new ByteArrayInputStream(text.getBytes(), beginIdx, text.length() - beginIdx);
        try {
            BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream), 5000);
            String line = "";
            for (int i = 0; (line = reader.readLine()) != null; i++) {
                if (i == 0) { // process the declaration line
                    // extract 'id' property
                    mpAnalyzer = findAnalyzer(line);
                    if (mpAnalyzer == null) {
                        String msg = MPAnalyzerLabel.getString(MPAnalyzerLabel._UNKNOWN_MICRO_PATTERN, line);
                        throw new MPException(MPException._UNKNOWN_MICRO_PATTERN, msg);
                    }

                    //if ("DCF".equals(mpAnalyzer.getIdentifier()))
                    //    System.out.println("");
                    // get 'dot' property 
                    boolean hasDot = false;
                    if (line.length() > IMPAnalyzer._RIGHT_MARGIN) {
                        String rightMargin = line.substring(IMPAnalyzer._RIGHT_MARGIN).toLowerCase();
                        if (rightMargin.indexOf(_DOT) >= 0)
                            hasDot = true;
                    }

                    // get concrete identifier (only for generic pattern)
                    String concreteIdentifier = mpAnalyzer.getIdentifier();
                    if (mpAnalyzer.getIdentifier().endsWith("*")) {
                        int start = IMPAnalyzer._STAR_COLUMN + IMPAnalyzer._MP_DECLARATION_TOKEN.length();
                        int end = line.indexOf(' ', start);
                        if (end == -1)
                            end = text.length();
                        concreteIdentifier = line.substring(start, end);
                    }

                    mpAnalyzerResult = new MPAnalyzerResult(mpAnalyzer.getIdentifier(), concreteIdentifier, hasDot);
                    int start = IMPAnalyzer._STAR_COLUMN + IMPAnalyzer._MP_DECLARATION_TOKEN.length() + concreteIdentifier.length();
                    int end = Math.min(IMPAnalyzer._RIGHT_MARGIN, line.length());
                    String truncatedLine = line.substring(start, end);

                    if (parseLine(truncatedLine, allPairs)) {
                        headerBuffer.append(line).append(lineSeparator);
                        parameterBuffer = new StringBuilder(truncatedLine);
                    }
                    else {
                        String msg = MPAnalyzerLabel.getString(MPAnalyzerLabel._INVALID_SEQUENCE, mpAnalyzer.getIdentifier(), line);
                        throw new MPException(MPException._INVALID_SEQUENCE, msg);
                    }
                }
                else { // process the next lines
                    if (line.indexOf(IMPAnalyzer._MP_NEXT_LINE_TOKEN) == IMPAnalyzer._STAR_COLUMN) { // new syntax next line is recognized
                        int start = IMPAnalyzer._STAR_COLUMN + IMPAnalyzer._MP_NEXT_LINE_TOKEN.length();
                        int end = Math.min(IMPAnalyzer._RIGHT_MARGIN, line.length());
                        String truncatedLine = line.substring(start, end);

                        if (parseLine(truncatedLine, allPairs)) {
                            headerBuffer.append(line).append(lineSeparator);
                            parameterBuffer.append(truncatedLine);
                        }
                        else {
                            String msg = MPAnalyzerLabel.getString(MPAnalyzerLabel._INVALID_SEQUENCE,
                                mpAnalyzer.getIdentifier(), line );
                            throw new MPException(MPException._INVALID_SEQUENCE, msg);
                        }
                    }
                    else {
                        // specific statement for WF micro-pattern to catch 2nd line (no other solution! ... beurk!)
                        if ("WF".equals(mpAnalyzer.getIdentifier()) && line.indexOf(IMPAnalyzer._STAR) == IMPAnalyzer._STAR_COLUMN && i == 1) {
                            String truncatedLine = line.substring(IMPAnalyzer._STAR_COLUMN + IMPAnalyzer._STAR.length());

                            if (truncatedLine.trim().length() > 0) {
                                if (parseLine(truncatedLine, allPairs)) {
                                    mpAnalyzerResult.setNeedFormat(true);
                                    headerBuffer.append(line).append(lineSeparator);
                                    parameterBuffer.append(truncatedLine);
                                }
                            }
                        }

                        break;
                    }
                }
            }

            reader.close();
            inputStream.reset();
            inputStream.close();
        } catch (IOException e) {
        }

        // check the micro-pattern parameters syntax and build the map of micro-pattern parameters
        if (mpAnalyzer != null) {
            mpAnalyzerResult.setHeader(headerBuffer.toString());
            List<String> malformedMessages = new ArrayList<String>();
            Map<String, Object> parameters = acceptParameters(mpAnalyzer, allPairs, lineSeparator, context, malformedMessages);

            if (parameters != null)
                mpAnalyzerResult.getParameters().putAll(parameters);
            else
                mpAnalyzerResult.getMalformedMessages().addAll(malformedMessages);
        }

        return mpAnalyzerResult;
    }


    public static IAnalyzerResult analyze2( String text, int beginIdx, IMicroPatternProcessingContext context )
    		throws MPException
    {
        int eolIdx = Util.eolIndex( text, beginIdx );
        int nextLineIdx = Util.nextLineIndex( text, eolIdx );
        String lineSeparator = (eolIdx < nextLineIdx ? text.substring( eolIdx, nextLineIdx ) : Util.EOL);
        String line = text.substring( beginIdx, eolIdx );

        // extract 'id' property
    	IMPAnalyzer mpAnalyzer = findAnalyzer( line );
        if ( mpAnalyzer == null )
            throw new MPException(
            	MPException._UNKNOWN_MICRO_PATTERN,
            	MPAnalyzerLabel.getString( MPAnalyzerLabel._UNKNOWN_MICRO_PATTERN, line ) );

        // get 'dot' property 
        boolean hasDot = line.length() > IMPAnalyzer._RIGHT_MARGIN
        	&& line.substring( IMPAnalyzer._RIGHT_MARGIN ).trim().equalsIgnoreCase( _DOT );

        // get concrete identifier (only for generic pattern)
        String identifier = mpAnalyzer.getIdentifier(), concreteIdentifier = identifier;
        if ( identifier.endsWith( "*" ) )
        {
            int start = IMPAnalyzer._STAR_COLUMN + IMPAnalyzer._MP_DECLARATION_TOKEN.length();
            int end = line.indexOf( ' ', start );
            concreteIdentifier = line.substring( start, end >= 0 ? end : line.length() );
        }

        IAnalyzerResult mpAnalyzerResult = new MPAnalyzerResult( identifier, concreteIdentifier, hasDot );
        int start = IMPAnalyzer._STAR_COLUMN + IMPAnalyzer._MP_DECLARATION_TOKEN.length() + concreteIdentifier.length();
        int end = Math.min( IMPAnalyzer._RIGHT_MARGIN, line.length() );
        LinkedHashMap<String, String> allPairs = new LinkedHashMap<String,String>();
        if ( !parseLine( line.substring( start, end ), allPairs ) )
            throw new MPException(
               	MPException._INVALID_SEQUENCE,
               	MPAnalyzerLabel.getString( MPAnalyzerLabel._INVALID_SEQUENCE, identifier, line ) );

        StringBuilder headerBuffer = new StringBuilder( line ).append( lineSeparator );
        int idx = nextLineIdx;
        eolIdx = Util.eolIndex( text, idx );
        nextLineIdx = Util.nextLineIndex( text, eolIdx );
        line = text.substring( idx, eolIdx );

        // specific statement for WF micro-pattern to catch 2nd line (no other solution! ... beurk!)
        if ( "WF".equals( identifier ) && line.startsWith( IMPAnalyzer._STAR, IMPAnalyzer._STAR_COLUMN ) && !line.startsWith( IMPAnalyzer._MP_NEXT_LINE_TOKEN, IMPAnalyzer._STAR_COLUMN ) )
        {
            String truncatedLine = line.substring( IMPAnalyzer._STAR_COLUMN + IMPAnalyzer._STAR.length() ).trim();
            if ( truncatedLine.length() > 0 && parseLine( truncatedLine, allPairs ) )
            {
                mpAnalyzerResult.setNeedFormat( true );
                headerBuffer.append( line ).append( lineSeparator );
            }
        }
        else while ( line.startsWith( IMPAnalyzer._MP_NEXT_LINE_TOKEN, IMPAnalyzer._STAR_COLUMN ) )
        {
            // process the next lines
            start = IMPAnalyzer._STAR_COLUMN + IMPAnalyzer._MP_NEXT_LINE_TOKEN.length();
            end = Math.min( IMPAnalyzer._RIGHT_MARGIN, line.length() );
            if ( !parseLine( line.substring( start, end ), allPairs ) )
                throw new MPException(
                	MPException._INVALID_SEQUENCE,
                	MPAnalyzerLabel.getString( MPAnalyzerLabel._INVALID_SEQUENCE, identifier, line ) );

            headerBuffer.append( line ).append( lineSeparator );
            idx = nextLineIdx;
            eolIdx = Util.eolIndex( text, idx );
            nextLineIdx = Util.nextLineIndex( text, eolIdx );
            line = text.substring( idx, eolIdx );
        }

        // check the micro-pattern parameters syntax and build the map of micro-pattern parameters
        mpAnalyzerResult.setHeader( headerBuffer.toString() );
        ArrayList<String> malformedMessages = new ArrayList<String>();
        Map<String,Object> parameters = acceptParameters( mpAnalyzer, allPairs, lineSeparator, context, malformedMessages );
        if ( parameters != null )
            mpAnalyzerResult.getParameters().putAll( parameters );
        else
            mpAnalyzerResult.getMalformedMessages().addAll( malformedMessages );

        return mpAnalyzerResult;
    }

 
    /**
     * Indicates if the micropattern identifier is recognized by an analyzer (an able to update its declaration with a new syntax).
     * Generic micropatterns (i.e. X* or Y* analyzer identifier) are recognized with "XNN" or with "XAA" parameter.
     */
    public static boolean accept(String microPatternId) {
        // step1: standard analyzers are examined
        if (getMPAnalyzerImplementors().containsKey(microPatternId))
            return true;

        // step2: generic analyzers are examined
        for (IMPAnalyzer mpAnalyzer : getMPAnalyzerImplementors().values()) {
            if (mpAnalyzer.getIdentifier().endsWith("*")) {
                String mpAnalyzerPrefix = mpAnalyzer.getIdentifier().substring(0, mpAnalyzer.getIdentifier().length() - 1);
                if (microPatternId.startsWith(mpAnalyzerPrefix))
                    return true;
            }
        }

        return false;
    }

    /**
     * Only for the JUnit plugin. Test the line parsing of the micropattern new syntax.
     */
    public static boolean testLineParser(String line, Map<String, String> allPairs) {
        return parseLine(line, allPairs);
    }

    /**
     * Indicates if the 'line' parameter contains a list of key=value pairs.
     */
    private static boolean parseLine(String line, Map<String, String> allPairs) {
        line = line.trim();
        if (line.length() == 0)
            return true;

        // remove blanks before and after the '=' car.
        //line = removeBlanks(line); // commented because WF parameter SEL can ends with '='

        if (line.charAt(0) == '*' || line.charAt(0) == '=' || line.charAt(0) == '"') // line beginning with these cars. are considered not parsable
            return false;

        // split on the ' ' car. to isolate key-value pair
        Map<String, String> pairs = new LinkedHashMap<String, String>();
        List<String> ls_words = splitLine(line);
        for (String word : ls_words) {
            int index = word.indexOf('=');
            if (index > 0) {
                String key = word.substring(0, index);
                String value = word.substring(index + 1);

                // if key is empty, the parsing failed
                if (key.length() == 0)
                    return false;

                // if key begin with car. '"' or ''', parsing failed
                if (key.charAt(0) == '\'' || key.charAt(0) == '"')
                    return false;

                // key ends with blank, or value starts with blank, parsing failed
                if (key.endsWith(" ") || value.startsWith(" "))
                    return false;

                // key contains blanks, the parsing failed
                if (key.contains(" "))
                    return false;

                // if the value contains a double-quote in first and last position, the double-quotes are removed
                if (value.length() > 1 && value.charAt(0) == '"' && value.charAt(value.length() - 1) == '"') {
                    value = value.substring(1, value.length() - 1);
                }

                pairs.put(key, value);
            }
            else
                return false;
        }

        allPairs.putAll(pairs);
        return true;
    }

    /**
     * Indicates if the 'pairs' describes a valid set of key-value pair, according to the micro-pattern implementation.
     * Return a map of key-value pair if the syntax is correct (value instance of String, Integer, Boolean or MPReference),
     * Return null, and feed the 'malformed messages' parameter otherwise.
     */
    private static Map<String, Object> acceptParameters(IMPAnalyzer mpAnalyzer, Map<String, String> allPairs, String lineSeparator,
            IMicroPatternProcessingContext context, List<String> malformedMessages) throws MPException {
        Map<String, MPParameter> allParameters = new HashMap<String, MPParameter>();
        allParameters.putAll(mpAnalyzer.getKeyedParameters()); // collect common parameters
        if (mpAnalyzer.getKeyedPatterns().size() > 0) {
            if (context != null) { // collect specific parameters of all patterns
                String patternName = context.getGenerationProperties().getProperty("PATTERN_NAME");
                MPPattern mpPattern = mpAnalyzer.getKeyedPatterns().get(patternName);
                if (mpPattern != null) {
                    // add specific parameters (or replace existing)
                    allParameters.putAll(mpPattern.getKeyedParameters());
                }
            }
            else { // collect specific parameters for the current pattern
                for (MPPattern mpPattern : mpAnalyzer.getKeyedPatterns().values()) {
                    // add specific parameters (or replace existing)
                    allParameters.putAll(mpPattern.getKeyedParameters());
                }
            }
        }

        // check the parameter names
        for (String pairKey : allPairs.keySet()) {
            if (!allParameters.containsKey(pairKey)) {
                malformedMessages.add(
                	MPAnalyzerLabel.getString(MPAnalyzerLabel._UNKNOWN_PARAMETER, mpAnalyzer.getIdentifier(), pairKey));
            }
        }

        // check the required parameters
        for (MPParameter mpParameter : allParameters.values()) {
            if (context == null || mpParameter.isOptional()) // context null => all parameters are considered optional
                continue;

            if (!allPairs.containsKey(mpParameter.getName())) { // a required parameter is missing
                malformedMessages.add(
                	MPAnalyzerLabel.getString(
                		MPAnalyzerLabel._MISSING_PARAMETER, mpAnalyzer.getIdentifier(), mpParameter.getName()));
            }
            else {
                String pairValue = allPairs.get(mpParameter.getName());
                if (pairValue == null || pairValue.length() == 0) { // a required parameter is empty
                    malformedMessages.add(
                        MPAnalyzerLabel.getString(
                            MPAnalyzerLabel._MISSING_PARAMETER, mpAnalyzer.getIdentifier(), mpParameter.getName()));
                }
            }
        }

        // at this step, if unknown or missing parameters, null is returned
        if (malformedMessages.size() > 0)
            return null;

        // check the parameters type, and feed the map of casted parameters
        boolean accept = true;
        Map<String, Object> castedParameters = new LinkedHashMap<String, Object>();
        for (Map.Entry<String, String> pair : allPairs.entrySet()) {
            String pairKey = pair.getKey();
            String pairValue = pair.getValue();

            MPParameter mpParameter = allParameters.get(pairKey);
            if (mpParameter != null && mpAnalyzer.isValueCheckable(pairValue)) { // recognized parameter, and pairValue checkable
                if (IMPAnalyzer._TYPE_STRING.equals(mpParameter.getType())) {
                    castedParameters.put(pairKey, pairValue);
                }
                else if (IMPAnalyzer._TYPE_NUMERIC.equals(mpParameter.getType())) {
                    try {
                        Integer intValue = Integer.parseInt(pairValue);
                        castedParameters.put(pairKey, intValue);
                    } catch (NumberFormatException e) {
                        malformedMessages.add(
                            MPAnalyzerLabel.getString(
                                MPAnalyzerLabel._NUMERIC_PARAMETER, mpAnalyzer.getIdentifier(), pairKey ));
                        accept = false;
                    }
                }
                else if (IMPAnalyzer._TYPE_BOOLEAN.equals(mpParameter.getType())) {
                    if (Boolean.toString(true).equalsIgnoreCase(pairValue) || Boolean.toString(false).equalsIgnoreCase(pairValue)) {
                        Boolean booleanValue = Boolean.parseBoolean(pairValue);
                        castedParameters.put(pairKey, booleanValue);
                    }
                    else {
                        malformedMessages.add(
                        	MPAnalyzerLabel.getString(
                        		MPAnalyzerLabel._BOOLEAN_PARAMETER, mpAnalyzer.getIdentifier(), pairKey));
                        accept = false;
                    }
                }
                else if (IMPAnalyzer._TYPE_DESIGN.equals(mpParameter.getType())) {
                    MPDesignReference designRef = new MPDesignReference(pairValue, mpParameter.getDesignType(), mpAnalyzer.getIdentifier());
                    castedParameters.put(pairKey, designRef);
                }
            }
            else { // not recognized parameter or pairVvalue not checkable (i.e macro parameter in a pacbase context)
                castedParameters.put(pairKey, pairValue);
            }
        }

        // the map of casted parameters is checked by each micro-pattern analyzer
        accept = accept && mpAnalyzer.acceptParameters(castedParameters, malformedMessages);
        if (context != null) {
            IMPExtendedAnalyzer mpExtendedAnalyzer = getMPExtendedAnalyzerImplementors().get(mpAnalyzer.getIdentifier());
            if (mpExtendedAnalyzer != null)
                accept = accept && mpExtendedAnalyzer.acceptParameters(castedParameters, lineSeparator, context, malformedMessages);
        }

        if (!accept || malformedMessages.size() > 0)
            castedParameters = null;

        return castedParameters;
    }

    /**
     * Build the string 'parameterLine' with the following pattern:
     * . each parameter respect the pattern 'param name'='param value',
     * . a blank car. separates each parameter.
     */
    private static String removeBlanks(String parameterLine) {
        StringBuilder buffer1 = new StringBuilder();
        int state = 0;

        // remove blanks after each car. '='.
        for (int i = 0; i < parameterLine.length(); i++) {
            char c = parameterLine.charAt(i);
            if (c == '=')
                state = 1;
            else if (c == ' ') {
                if (state == 1)
                    continue;
            }
            else
                state = 0;

            buffer1.append(c);
        }

        // remove blanks before each car. '='.
        buffer1 = buffer1.reverse();
        StringBuilder buffer2 = new StringBuilder();
        for (int i = 0; i < buffer1.length(); i++) {
            char c = buffer1.charAt(i);
            if (c == '=')
                state = 1;
            else if (c == ' ') {
                if (state == 1)
                    continue;
            }
            else
                state = 0;

            buffer2.append(c);
        }
        parameterLine = buffer2.reverse().toString();

        return parameterLine;
    }

    /**
     * Split the line on the car. ' ' to isolate key=value pairs.
     * Value can contains blank if it is framed by single-quote or by double-quote.
     * TAKE CARE: The parameter 'line' contains a MP old syntax or a MP new syntax.
     *            This method must be modified carefully.
     *            The JUnit TestLineParser must be used to avoid regression.
     */
    private static List<String> splitLine(String line) {
        // split on the car. ' ', but preserve sequences between simple-quote or double-quote
        List<String> ls_words = new ArrayList<String>();
        StringBuilder word = new StringBuilder("");
        boolean singleQuote = false;
        boolean doubleQuote = false;
        for (int i = 0; i < line.length(); i++) {
            char c = line.charAt(i);

            if (c == ' ') {
                if (!singleQuote && !doubleQuote) {
                    if (word.length() > 0) {
                        ls_words.add(word.toString());
                        word = new StringBuilder("");
                    }
                }
                else
                    word.append(c);
            }
            else if (c == '\'') {
                if (!doubleQuote) {
                    if (singleQuote) {
                        if (i == line.length() - 1) // end of line
                            singleQuote = false; // closing single-quote
                        else if (line.charAt(i + 1) == ' ' && word.length() > 0) // test word length since WI 34081
                            singleQuote = false; // closing single-quote
                        else
                            word.append(c);
                    }
                    else
                        singleQuote = true; // opening single-quote
                }
                else
                    word.append(c);
            }
            else if (c == '"') {
                if (!singleQuote) {
                    if (doubleQuote) {
                        if (i == line.length() - 1) // end of line
                            doubleQuote = false; // closing double-quote
                        else if (line.charAt(i + 1) == ' ' && word.length() > 0) // test word length since WI 34081
                            doubleQuote = false; // closing double-quote
                        else
                            word.append(c);
                    }
                    else
                        doubleQuote = true; // opening double-quote
                }
                else
                    word.append(c);
            }
            else {
                word.append(c);
            }
        }

        if (word.length() > 0)
            ls_words.add(word.toString());

        return ls_words;
    }
}
