/*
 * Decompiled with CFR 0.152.
 */
package com.ibm.pdp.engine.turbo.core;

import com.ibm.pdp.engine.extension.DefaultTextPartitioner;
import com.ibm.pdp.engine.extension.ITextPartitioner;
import com.ibm.pdp.engine.turbo.core.BasicTextPartition;
import com.ibm.pdp.engine.turbo.core.Dictionary;
import com.ibm.pdp.engine.turbo.core.ISubTextPartition;
import com.ibm.pdp.engine.turbo.core.ITextPartition;
import com.ibm.pdp.engine.turbo.core.IncrementalTextPartition;
import com.ibm.pdp.util.Ints;
import com.ibm.pdp.util.RandomBuilder;
import com.ibm.pdp.util.Strings;
import com.ibm.pdp.util.ints.IntSequence;
import java.util.Formatter;

public class TextPartitionTest {
    protected static int testCount;
    protected static RandomBuilder random;
    protected static Formatter fmt;
    protected static final CharSequence initialText;
    protected static int checkCount;
    public static final String copyright = "Licensed Materials - Property of IBM\n5725-H03\n(C) Copyright IBM Corp. 2010, 2014.   All rights reserved.\nUS Government Users Restricted Rights - Use, duplication or disclosure restricted by GSA ADP Schedule Contract with IBM Corp.";

    static {
        initialText = "/*\n * Source File Name : MergeIterator.java\n * IBM Confidential\n * OCO Source Materials\n * 5655-F37\n * (C) Copyright IBM Corp. 1983, 2005\n * The source code for this program is not published or otherwise divested of its trade secrets,\n * irrespective of what has been deposited with the U. S. Copyright Office.\n */\npackage com.ibm.pdp.util.containers;\n\nimport java.util.*;\n\nimport com.ibm.pdp.util.sort.NaturalComparator;\n\n/**\n * It\u00e9rateur capable de fusionner plusieurs it\u00e9rateurs tri\u00e9s selon le m\u00eame crit\u00e8re.\n * <p>\n * Cet it\u00e9rateur doit \u00eatre initialis\u00e9 avec une liste d'it\u00e9rateurs dont les \u00e9l\u00e9ments sont tri\u00e9s selon le m\u00eame crit\u00e8re\n * (m\u00eame <tt>Comparator</tt>). Ensuite, les m\u00e9thodes <tt>hasNext</tt> et <tt>next</tt> permettent de parcourir\n * l'ensemble des \u00e9l\u00e9ments de tous les it\u00e9rateurs donn\u00e9s, dans l'ordre induit par le comparateur.\n * <p>\n * La m\u00e9thode <tt>remove</tt> de l'it\u00e9rateur de fusion permet de supprimer, de la collection de laquelle il a \u00e9t\u00e9\n * extrait, l'\u00e9l\u00e9ment qui vient d'\u00eatre rendu.\n * <p>\n * D'autre part, un indicateur <tt>skipSameElements</tt> permet d'ignorer ou non les \u00e9l\u00e9ments multiples rendus par\n * plusieurs it\u00e9rateurs.\n * <p>\n * Exemple, soit deux it\u00e9rateurs d'entiers tri\u00e9s rendant les entiers suivants :\n * <code><pre>\n * Iter1 : 4 6 7 9\n * Iter2 : 1 2 3 6 7 8</pre></code>\n * <p>\n * La fusion donnera l'it\u00e9rateur :\n * <code><pre>\n * Iter1 :       4 6   7     9\n * Iter2 : 1 2 3 | | 6 | 7 8 |\n *         | | | | | | | | | |\n * Fusion: 1 2 3 4 6 6 7 7 8 9</pre></code>\n * <p>\n * L'algorithme consiste simplement \u00e0 demander \u00e0 chaque it\u00e9rateur son prochain \u00e9l\u00e9ment et rendre le plus petit (selon\n * le comparateur fourni).\n * <p>\n * En positionnant l'indicateur <tt>skipSameElements</tt> \u00e0 vrai, on peut cacher les \u00e9l\u00e9ments redondants.\n * Dans ce cas, la fusion donnera :\n * <code><pre>\n * Iter1 :       4 6   7     9\n * Iter2 : 1 2 3 | | 6 | 7 8 |\n *         | | | | |   |   | |\n * Fusion: 1 2 3 4 6   7   8 9</pre></code>\n * <p> \n * On voit que les valeurs 6 et 7 de l'it\u00e9rateur 2 ne sont pas rendues, car d\u00e9j\u00e0 rendues en tant qu'\u00e9l\u00e9ments de\n * l'it\u00e9rateur 1. La r\u00e8gle est que l'it\u00e9rateur qui apparait avant un autre dans la liste des it\u00e9rateurs fournis est\n * prioritaire par rapport \u00e0 l'autre lorsque les valeurs \u00e0 rendre sont \u00e9gales.\n * <p>\n * En revanche, si un it\u00e9rateur fourni au d\u00e9part poss\u00e8de d\u00e9j\u00e0 des \u00e9l\u00e9ments multiples, ceux-ci ne seront pas d\u00e9tect\u00e9s et\n * l'it\u00e9rateur obtenu poss\u00e8dera aussi des \u00e9l\u00e9ments multiples, m\u00eame lorsque <tt>skipSameElements</tt> est vrai.\n * Par exemple, avec <tt>skipSameElements</tt> \u00e0 vrai :\n * <code><pre>\n * Iter1 :         4 4 6     7       9 9\n * Iter2 : 1 2 2 3 | | | 6 6 | 7 7 8 | |\n *         | | | | | | |   | |   | | | |\n * Fusion: 1 2 2 3 4 4 6   6 7   7 8 9 9</pre></code>\n * <p>\n * En positionnant l'indicateur <tt>skipSameElements</tt> \u00e0 faux, tous les \u00e9l\u00e9ments sont rendus, la fusion donnera donc:\n * <code><pre>\n * Iter1 :         4 4 6     7       9 9\n * Iter2 : 1 2 2 3 | | | 6 6 | 7 7 8 | |\n *         | | | | | | | | | | | | | | |\n * Fusion: 1 2 2 3 4 4 6 6 6 7 7 7 8 9 9</pre></code>\n * <p>\n * Cela dit, si n\u00e9cessaire, cacher les doubles dans un it\u00e9rateur tri\u00e9 est assez simple, par exemple en utilisant un\n * <tt>FilterIterator</tt> : \n * <code><pre>\n * final Comparator cmp = ...\n * Iterator iterWithMultipleValues = ...\n * Iterator iterWithoutMultipleValues =\n *     new FilterIterator( iterWithMultipleValues )\n *     {\n *         private boolean notFirst;\n *         private Object previous;\n *         public boolean accept( Object o )\n *         {\n *             if ( notFirst && cmp.compare( o, previous ) == 0 ) return false;\n *             previous = o;\n *             notFirst = true;\n *             return true;\n *         }\n *     };</pre></code>\n * <p>\n * Si un it\u00e9rateur propos\u00e9 au d\u00e9part n'est pas correctement tri\u00e9, aucune erreur ne sera report\u00e9e, mais l'it\u00e9rateur\n * obtenu ne le sera pas non plus. En fait, l'algorithme se d\u00e9roule normalement, ce qui donne par exemple :\n * <code><pre>\n * Iter1 :       6   4     9 7\n * Iter2 : 1 2 3 | 6 | 7 8 | |\n *         | | | | | | | | | |\n * Fusion: 1 2 3 6 6 4 7 8 9 7</pre></code>\n * <p>\n * En positionnant l'indicateur <tt>skipSameElements</tt> \u00e0 vrai, la fusion donnera :\n * <code><pre>\n * Iter1 :       6   4     9 7\n * Iter2 : 1 2 3 | 6 | 7 8 | |\n *         | | | |   | | | | |\n * Fusion: 1 2 3 6   4 7 8 9 7</pre></code>\n * <p>\n * Note : lorsque l'indicateur <tt>skipSameElements</tt> est vrai et qu'on applique <tt>remove</tt> a un \u00e9l\u00e9ment\n * qui est rendu par plusieurs it\u00e9rateurs, la suppression est transmise \u00e0 tous les it\u00e9rateurs ayant rendu cet \u00e9l\u00e9ment.\n * En revanche, lorque l'indicateur <tt>skipSameElement</tt> est faux, la suppression ne s'applique qu'\u00e0 l'it\u00e9rateur\n * ayant rendu le dernier \u00e9l\u00e9ment.\n * <p>\n * L'utilisation d'un it\u00e9rateur de fusion permet d'optimiser certains traitements en \u00e9vitant des copies ou des\n * parcours inutiles de collections.\n * <p> \n * Par exemple, on dispose de trois collections d'<tt>Employee</tt> diff\u00e9rentes, tri\u00e9es dans l'ordre croissant des\n * anciennet\u00e9s. On dispose aussi d'une m\u00e9thode\n * <tt>double[] averageSalaryBySeniority( Iterator employesIter, int yearsBracket )</tt>\n * permettant de calculer les salaires moyen des employ\u00e9s par tranche d'anciennet\u00e9.\n * <p>\n * On souhaite obtenir les moyennes des salaires de l'ensemble des employ\u00e9s (trois collections r\u00e9unies) par tranche\n * de 5 ans d'anciennet\u00e9, en utilisant la m\u00e9thode <tt>averageSalaryBySeniority</tt>. On doit donc passer \u00e0 la m\u00e9thode\n * <tt>averageSalaryBySeniority</tt> un it\u00e9rateur tri\u00e9 par anciennet\u00e9 et parcourant les trois collections.\n * <p><code><pre>\n * SortedSet employees1 = ...;\n * SortedSet employees2 = ...;\n * SortedSet employees3 = ...;\n *\n * Iterator employees =\n *     new MergeIterator(\n *         employees1.iterator(), employees2.iterator(), employees3.iterator(),\n *         employees1.comparator() );\n *\n * double[] avgSalaryByFiveYearsSeniority = averageSalaryBySeniority( employees, 5 );</pre></code>\n * <p>\n * Sans it\u00e9rateur de fusion, on aurait vraissemblablement \u00e9crit quelque chose comme cela :\n * <p><code><pre>\n * SortedSet employees1 = ...;\n * SortedSet employees2 = ...;\n * SortedSet employees3 = ...;\n *\n * SortedSet employees = new TreeSet( employees1.comparator() );\n * employees.addAll( employees1 );\n * employees.addAll( employees2 );\n * employees.addAll( employees3 );\n *\n * double[] avgSalaryByFiveYearsSeniority = averageSalaryBySeniority( employees.iterator(), 5 );</pre></code>\n * <p>\n * On voit que dans ce cas, l'utilisation d'un it\u00e9rateur de fusion permet d'\u00e9viter un parcours pr\u00e9alable\n * des collections et une copie des \u00e9l\u00e9ments \u00e0 traiter dans une collection temporaire.\n * <p>\n * N\u00e9anmoins, si une m\u00eame fusion devait \u00eatre reconstruite plusieurs fois, la technique consistant \u00e0 cr\u00e9er une\n * collection temporaire puis de l'it\u00e9rer plusieurs fois pourrait s'av\u00e9rer plus efficace.\n *\n */\n@SuppressWarnings(\"unchecked\")\npublic class MergeIterator<E> implements Iterator<E>\n{\n\t// Liste contenant les it\u00e9rateurs \u00e0 parcourir (non encore pris en compte dans la fusion).\n\tprotected List<Iterator<E>> newIterators;\n\n\t// Comparateur \u00e0 utiliser, qui doit \u00eatre compatible avec l'ordre des \u00e9l\u00e9ments dans chaque it\u00e9rateur.\n\tprotected Comparator<? super E> cmp;\n\n\t// Indique si des \u00e9l\u00e9ments identiques (d'apr\u00e8s le comparateur utilis\u00e9) rendus par les it\u00e9rateurs doivent \u00eatre tous\n\t// montr\u00e9s ou non. Si les \"multiples\" sont cach\u00e9s, un seul repr\u00e9sentant des \u00e9l\u00e9ments \u00e9gaux sera rendu par\n\t// l'it\u00e9rateur.\n\tprotected boolean skipSameElements;\n\n\t// Ensemble tri\u00e9 utilis\u00e9 pendant le parcours.\n\t// Contient des SortableIterator tri\u00e9s en fonction d'un SortableIteratorComparator.\n\tprotected SortedSet<SortableIterator<E>> sortedIterators;\n\n\t// Liste contenant les it\u00e9rateurs s\u00e9lectionn\u00e9s pour rendre le prochain \u00e9l\u00e9ment.\n\tprotected List<SortableIterator<E>> selectedIterators;\n\n\t// Compteur utilis\u00e9 pour ordonner les \u00e9l\u00e9ments \u00e9gaux rendus par les it\u00e9rateurs.\n\tprotected int iterCount;\n\n\n\t/**\n\t * Cr\u00e9ation d'un it\u00e9rateur de fusion s'appuyant sur la comparaison naturelle des \u00e9l\u00e9ments (voir interface\n\t * <tt>java.util.Comparable</tt>).\n\t */\n\tpublic MergeIterator()\n\t{\n\t\tsuper();\n\t\tcmp = NaturalComparator.DEFAULT_COMPARATOR;\n\t}\n\n\n\t/**\n\t * Cr\u00e9ation d'un it\u00e9rateur de fusion s'appuyant sur le comparateur fourni.\n\t */\n\tpublic MergeIterator( Comparator<? super E> comparator )\n\t{\n\t\tsuper();\n\t\tcmp = comparator;\n\t}\n\n\n\t/**\n\t * Cr\u00e9ation d'un it\u00e9rateur de fusion s'appuyant sur la comparaison naturelle des \u00e9l\u00e9ments (voir interface\n\t * <tt>java.util.Comparable</tt>).\n\t *\n\t * @param iterators les it\u00e9rateurs participant \u00e0 la fusion.\n\t */\n\tpublic MergeIterator( Iterator<? extends E>... iterators )\n\t{\n\t\tthis();\n\t\tnewIterators = newIteratorsList();\n\t\tfor ( Iterator<? extends E> iter : iterators )\n\t\t\tnewIterators.add( (Iterator<E>)iter );\n\t}\n\n\n\t/**\n\t * Cr\u00e9ation d'un it\u00e9rateur de fusion s'appuyant sur un comparateur.\n\t *\n\t * @param iterators les it\u00e9rateurs participant \u00e0 la fusion.\n\t */\n\tpublic MergeIterator( Comparator<? super E> comparator, Iterator<? extends E>... iterators )\n\t{\n\t\tthis( comparator );\n\t\tnewIterators = newIteratorsList();\n\t\tfor ( Iterator<? extends E> iter : iterators )\n\t\t\tnewIterators.add( (Iterator<E>)iter );\n\t}\n\n\n\t/**\n\t * Cr\u00e9ation d'un it\u00e9rateur de fusion s'appuyant sur la comparaison naturelle des \u00e9l\u00e9ments (voir interface\n\t * <tt>java.util.Comparable</tt>).\n\t *\n\t * @param first premier it\u00e9rateur participant \u00e0 la fusion.\n\t * @param second second it\u00e9rateur participant \u00e0 la fusion.\n\t */\n\tpublic MergeIterator( Iterator<? extends E> first, Iterator<? extends E> second )\n\t{\n\t\tthis();\n\t\tnewIterators = newIteratorsList();\n\t\tnewIterators.add( (Iterator<E>)first );\n\t\tnewIterators.add( (Iterator<E>)second );\n\t}\n\n\n\t/**\n\t * Cr\u00e9ation d'un it\u00e9rateur de fusion s'appuyant sur la comparaison naturelle des \u00e9l\u00e9ments (voir interface\n\t * <tt>java.util.Comparable</tt>).\n\t *\n\t * @param first premier it\u00e9rateur participant \u00e0 la fusion.\n\t * @param second second it\u00e9rateur participant \u00e0 la fusion.\n\t * @param third troisi\u00e8me it\u00e9rateur participant \u00e0 la fusion.\n\t */\n\tpublic MergeIterator( Iterator<? extends E> first, Iterator<? extends E> second, Iterator<? extends E> third )\n\t{\n\t\tthis();\n\t\tnewIterators = newIteratorsList();\n\t\tnewIterators.add( (Iterator<E>)first );\n\t\tnewIterators.add( (Iterator<E>)second );\n\t\tnewIterators.add( (Iterator<E>)third );\n\t}\n\n\n\t/**\n\t * Cr\u00e9ation d'un it\u00e9rateur de fusion s'appuyant sur la comparaison naturelle des \u00e9l\u00e9ments (voir interface\n\t * <tt>java.util.Comparable</tt>).\n\t *\n\t * @param first premier it\u00e9rateur participant \u00e0 la fusion.\n\t * @param second second it\u00e9rateur participant \u00e0 la fusion.\n\t * @param third troisi\u00e8me it\u00e9rateur participant \u00e0 la fusion.\n\t * @param fourth quatri\u00e8me it\u00e9rateur participant \u00e0 la fusion.\n\t */\n\tpublic MergeIterator(\n\t\tIterator<? extends E> first, Iterator<? extends E> second,\n\t\tIterator<? extends E> third, Iterator<? extends E> fourth )\n\t{\n\t\tthis();\n\t\tnewIterators = newIteratorsList();\n\t\tnewIterators.add( (Iterator<E>)first );\n\t\tnewIterators.add( (Iterator<E>)second );\n\t\tnewIterators.add( (Iterator<E>)third );\n\t\tnewIterators.add( (Iterator<E>)fourth );\n\t}\n\n\n\t/**\n\t * Cr\u00e9ation d'un it\u00e9rateur de fusion s'appuyant sur le comparateur fourni.\n\t *\n\t * @param first premier it\u00e9rateur participant \u00e0 la fusion.\n\t * @param second second it\u00e9rateur participant \u00e0 la fusion.\n\t */\n\tpublic MergeIterator(\n\t\tIterator<? extends E> first, Iterator<? extends E> second, Comparator<? super E> comparator )\n\t{\n\t\tthis( comparator );\n\t\tnewIterators = newIteratorsList();\n\t\tnewIterators.add( (Iterator<E>)first );\n\t\tnewIterators.add( (Iterator<E>)second );\n\t}\n\n\n\t/**\n\t * Cr\u00e9ation d'un it\u00e9rateur de fusion s'appuyant sur le comparateur fourni.\n\t *\n\t * @param first premier it\u00e9rateur participant \u00e0 la fusion.\n\t * @param second second it\u00e9rateur participant \u00e0 la fusion.\n\t * @param third troisi\u00e8me it\u00e9rateur participant \u00e0 la fusion.\n\t */\n\tpublic MergeIterator(\n\t\tIterator<? extends E> first, Iterator<? extends E> second, Iterator<? extends E> third,\n\t\tComparator<? super E> comparator )\n\t{\n\t\tthis( comparator );\n\t\tnewIterators = newIteratorsList();\n\t\tnewIterators.add( (Iterator<E>)first );\n\t\tnewIterators.add( (Iterator<E>)second );\n\t\tnewIterators.add( (Iterator<E>)third );\n\t}\n\n\n\t/**\n\t * Cr\u00e9ation d'un it\u00e9rateur de fusion s'appuyant sur le comparateur fourni.\n\t *\n\t * @param first premier it\u00e9rateur participant \u00e0 la fusion.\n\t * @param second second it\u00e9rateur participant \u00e0 la fusion.\n\t * @param third troisi\u00e8me it\u00e9rateur participant \u00e0 la fusion.\n\t * @param fourth quatri\u00e8me it\u00e9rateur participant \u00e0 la fusion.\n\t */\n\tpublic MergeIterator(\n\t\tIterator<? extends E> first, Iterator<? extends E> second, Iterator<? extends E> third,\n\t\tIterator<? extends E> fourth, Comparator<? super E> comparator )\n\t{\n\t\tthis( comparator );\n\t\tnewIterators = newIteratorsList();\n\t\tnewIterators.add( (Iterator<E>)first );\n\t\tnewIterators.add( (Iterator<E>)second );\n\t\tnewIterators.add( (Iterator<E>)third );\n\t\tnewIterators.add( (Iterator<E>)fourth );\n\t}\n\n\n\t/**\n\t * Cr\u00e9ation d'un it\u00e9rateur de fusion s'appuyant sur le comparateur fourni.\n\t *\n\t * @param iteratorsList liste contenant les it\u00e9rateurs \u00e0 fusionner.\n\t */\n\tpublic MergeIterator( List<Iterator<? extends E>> iteratorsList, Comparator<? super E> comparator )\n\t{\n\t\tthis( comparator );\n\t\tnewIterators = (List)iteratorsList;\n\t}\n\n\n\t/**\n\t * Positionne la liste des it\u00e9rateurs \u00e0 fusionner.\n\t * <p>\n\t * Note : si le parcours a d\u00e9j\u00e0 commenc\u00e9, la liste d'it\u00e9rateurs fournie s'ajoute \u00e0 l'ensemble des\n\t * it\u00e9rateurs \u00e0 parcourir (\u00e0 partir du prochain appel \u00e0 <tt>hasNext</tt> ou <tt>next</tt>).\n\t *\n\t * @param iteratorsList La nouvelle liste d'it\u00e9rateurs \u00e0 parcourir.\n\t */\n\tpublic void setIterators( List<Iterator<? extends E>> iteratorsList )\n\t{\n\t\tnewIterators = (List)iteratorsList;\n\t}\n\n\n\t/**\n\t * Ajoute un it\u00e9rateur \u00e0 la liste des it\u00e9rateurs \u00e0 parcourir.\n\t * <p>\n\t * Un it\u00e9rateur peut \u00eatre ajout\u00e9 alors que le parcours \u00e0 d\u00e9j\u00e0 commenc\u00e9. Dans ce cas, les \u00e9l\u00e9ments du nouvel\n\t * it\u00e9rateur seront pris en compte lors du prochain appel \u00e0 <tt>hasNext</tt> ou <tt>next</tt>.\n\t *\n\t * @param iterator l'it\u00e9rateur \u00e0 ajouter.\n\t */\n\tpublic void addIterator( Iterator<? extends E> iterator )\n\t{\n\t\tif ( newIterators == null ) newIterators = newIteratorsList();\n\t\tnewIterators.add( (Iterator<E>)iterator );\n\t}\n\n\n\t/**\n\t * Rend le comparateur qui sera utilis\u00e9 pour d\u00e9terminer l'ordre des \u00e9l\u00e9ments rendus par cet it\u00e9rateur.\n\t * <p>\n\t * Les it\u00e9rateurs sur lequels cet it\u00e9rateur s'appuie sont suppos\u00e9s rendre leurs \u00e9l\u00e9ments dans un ordre compatible\n\t * avec ce comparateur.\n\t *\n\t * @return le comparateur utilis\u00e9 pour d\u00e9terminer l'ordre des \u00e9l\u00e9ments rendu par cet it\u00e9rateur.\n\t */\n\tpublic Comparator<? super E> getComparator() { return cmp; }\n\n\n\t/**\n\t * Positionne le comparateur \u00e0 utiliser pour d\u00e9terminer l'ordre des \u00e9l\u00e9ments rendus par cet it\u00e9rateur.\n\t * <p>\n\t * Par d\u00e9faut, le comparateur naturel des \u00e9l\u00e9ments sera utilis\u00e9 (<tt>NaturalComparator</tt>).\n\t * <p>\n\t * Le changement de comparateur alors que l'it\u00e9ration a d\u00e9j\u00e0 commenc\u00e9 induit un comportement non d\u00e9fini.\n\t *\n\t * @param comparator le nouveau comparateur \u00e0 utiliser.\n\t */\n\tpublic void setComparator( Comparator<? super E> comparator ) { cmp = comparator; }\n\n\n\t/**\n\t * Rend l'indicateur de parcours des \u00e9l\u00e9ments multiples.\n\t * <p>\n\t * Par d\u00e9faut, <tt>skipSameElements</tt> est faux, ce qui signifie que tous les \u00e9l\u00e9ments rendus par les\n\t * it\u00e9rateurs sous-jacents seront rendus par l'it\u00e9rateur fusionn\u00e9.\n\t * <p>\n\t * Lorsqu'il est positionn\u00e9 \u00e0 vrai, si le m\u00eame \u00e9l\u00e9ment (d'apr\u00e8s le comparateur fourni) est rendu par plusieurs des\n\t * it\u00e9rateurs fournis, un seul repr\u00e9sentant de l'\u00e9l\u00e9ment sera rendu par l'it\u00e9rateur de fusion. Le repr\u00e9sentant\n\t * choisi sera celui qui a \u00e9t\u00e9 rendu par l'it\u00e9rateur rencontr\u00e9 en premier dans la liste des it\u00e9rateurs fournis.\n\t *\n\t * @return\n\t * <tt>false</tt> si toutes les occurrences d'un m\u00eame \u00e9l\u00e9ment doivent \u00eatre rendues (valeur par d\u00e9faut),\n\t * <p>\n\t * <tt>true</tt> si un seul repr\u00e9sentant de chaque \u00e9l\u00e9ment multiple doit \u00eatre rendu.\n\t */\n\tpublic boolean getSkipSameElements() { return skipSameElements; }\n\n\n\t/**\n\t * Positionne l'indicateur de parcours des \u00e9l\u00e9ments multiples (voir <tt>getSkipSameElements</tt>).\n\t *\n\t * @param newValue nouvelle valeur de l'indicateur de parcours des \u00e9l\u00e9ments multiples.\n\t */\n\tpublic void setSkipSameElements( boolean newValue ) { skipSameElements = newValue; }\n\n\n\t/**\n\t * @return\n\t * <tt>true</tt> si au moins un des it\u00e9rateurs fournis poss\u00e8de encore un \u00e9l\u00e9ment \u00e0 rendre, <tt>false</tt> sinon.\n\t */\n\tpublic boolean hasNext()\n\t{\n\t\tif ( sortedIterators == null )\n\t\t{\n\t\t\t// Premier appel : initialisation des collections n\u00e9cessaires au parcours.\n\t\t\tsortedIterators = newIteratorsSortedSet( cmp );\n\t\t\tselectedIterators = newSortableIteratorsList();\n\t\t}\n\n\t\tif ( newIterators != null && !newIterators.isEmpty() )\n\t\t\t// S'il y a de nouveaux it\u00e9rateurs (typiquement, au premier appel), les prendre en compte.\n\t\t\tstoreNewIterators( newIterators );\n\n\t\treturn !sortedIterators.isEmpty() || (!selectedIterators.isEmpty() && hasNext( selectedIterators ));\n\t}\n\n\n\t/**\n\t * @return le prochain plus petit \u00e9l\u00e9ment parmis les suivants rendus par les chaque it\u00e9rateur fourni.\n\t */\n\tpublic E next()\n\t{\n\t\tif ( !hasNext() )\n\t\t\tthrow new NoSuchElementException( \"MergeIterator.next\" );;\n\n\t\tif ( !selectedIterators.isEmpty() )\n\t\t\tstoreIterators( selectedIterators );\n\n\t\treturn selectIterators();\n\t}\n\n\n\t/**\n\t * Suprime de la collection qui le contient, le dernier \u00e9l\u00e9ment rendu par cet it\u00e9rateur.\n\t * <p>\n\t * La suppression est transmise \u00e0 l'it\u00e9rateur sous-jacent ayant donn\u00e9 le dernier \u00e9l\u00e9ment rendu.\n\t * <p>\n\t * Dans le cas ou l'indicateur <tt>skipSameElements</tt> est vrai et que le dernier \u00e9l\u00e9ment rendu \u00e9tait rendu par\n\t * plusieurs it\u00e9rateurs sous-jacents, la suppression est transmise \u00e0 tous les it\u00e9rateurs qui ont rendu cet\n\t * \u00e9l\u00e9ment.\n\t */\n\tpublic void remove()\n\t{\n\t\tif ( selectedIterators == null || selectedIterators.isEmpty() )\n\t\t\tthrow new IllegalStateException( \"MergeIterator.remove\" );\n\n\t\t// Supprimer l'\u00e9l\u00e9ment courant sur chaque it\u00e9rateur s\u00e9lectionn\u00e9.\n\t\tIterator<SortableIterator<E>> iteratorsIter = selectedIterators.iterator();\n\t\twhile ( iteratorsIter.hasNext() ) iteratorsIter.next().remove();\n\n\t\t// Replacer les it\u00e9rateurs s\u00e9lectionn\u00e9s qui ont encore des \u00e9l\u00e9ments dans l'ensemble tri\u00e9.\n\t\tstoreIterators( selectedIterators );\n\t}\n\n\n\t/**\n\t * Contr\u00f4le de pr\u00e9sence d'au moins un \u00e9l\u00e9ment suivant dans une liste d'it\u00e9rateurs.\n\t *\n\t * @param iterators liste des it\u00e9rateurs \u00e0 interroger.\n\t * @return <tt>true</tt> si au moins un des it\u00e9rateurs pr\u00e9sents dans la liste poss\u00e8de encore\n\t * un \u00e9l\u00e9ment \u00e0 rendre, <tt>false</tt> sinon.\n\t */\n\tprotected boolean hasNext( List iterators )\n\t{\n\t\tIterator iteratorsIter = iterators.iterator();\n\t\twhile ( iteratorsIter.hasNext() )\n\t\t\tif ( ((Iterator)iteratorsIter.next()).hasNext() )\n\t\t\t\treturn true;\n\n\t\treturn false;\n\t}\n\n\n\t/**\n\t * Transf\u00e8re les it\u00e9rateurs qui se trouvent dans la liste pass\u00e9e en param\u00e8tre et qui ont encore des \u00e9l\u00e9ments \u00e0\n\t * rendre dans l'ensemble tri\u00e9 des it\u00e9rateurs <tt>sortedIterators</tt>.\n\t * <p>\n\t * Avant d'ajouter un it\u00e9rateur \u00e0 l'ensemble tri\u00e9, il est converti en <tt>SortableIterator</tt>\n\t * afin de supporter le tri.\n\t * <p>\n\t * En sortie, la liste d'it\u00e9rateurs pass\u00e9s en param\u00e8tres est vide.\n\t */\n\tprotected void storeNewIterators( List<Iterator<E>> iterators )\n\t{\n\t\t// Transfert des it\u00e9rateurs dans l'ensemble des it\u00e9rateurs tri\u00e9s.\n\t\tIterator<Iterator<E>> iteratorsIter = iterators.iterator();\n\t\twhile ( iteratorsIter.hasNext() )\n\t\t{\n\t\t\tIterator<E> iterator = iteratorsIter.next();\n\t\t\tif ( iterator.hasNext() )\n\t\t\t{\n\t\t\t\tSortableIterator<E> sortableIterator = newSortableIterator( iterator );\n\t\t\t\tsortableIterator.next();\n\t\t\t\tsortedIterators.add( sortableIterator );\n\t\t\t}\n\t\t}\n\n\t\t// Vidange des it\u00e9rateurs transf\u00e9r\u00e9s.\n\t\titerators.clear();\n\t}\n\n\n\t/**\n\t * Transf\u00e8re les it\u00e9rateurs qui se trouvent dans la liste pass\u00e9e en param\u00e8tre et qui ont encore des \u00e9l\u00e9ments \u00e0\n\t * rendre dans l'ensemble tri\u00e9 des it\u00e9rateurs <tt>sortedIterators</tt>.\n\t * <p>\n\t * En sortie, la liste d'it\u00e9rateurs pass\u00e9s en param\u00e8tres est vide.\n\t */\n\tprotected void storeIterators( List<SortableIterator<E>> iterators )\n\t{\n\t\t// Transfert des it\u00e9rateurs dans l'ensemble des it\u00e9rateurs tri\u00e9s.\n\t\tIterator<SortableIterator<E>> iteratorsIter = iterators.iterator();\n\t\twhile ( iteratorsIter.hasNext() )\n\t\t{\n\t\t\tSortableIterator<E> iterator = iteratorsIter.next();\n\t\t\tif ( iterator.hasNext() )\n\t\t\t{\n\t\t\t\titerator.next();\n\t\t\t\tsortedIterators.add( iterator );\n\t\t\t}\n\t\t}\n\n\t\t// Vidange des it\u00e9rateurs transf\u00e9r\u00e9s.\n\t\titerators.clear();\n\t}\n\n\n\t/**\n\t * Transf\u00e8re un ou plusieurs it\u00e9rateurs se trouvant en d\u00e9but de l'ensemble tri\u00e9 des it\u00e9rateurs\n\t * <tt>sortedIterators</tt> dans la liste des it\u00e9rateurs selectionn\u00e9s <tt>selectedIterators</tt>.\n\t * <p>\n\t * Le ou les it\u00e9rateurs ayant rendu le plus petit \u00e9l\u00e9ment sont supprim\u00e9s de l'ensemble tri\u00e9 et ajout\u00e9s \u00e0 la liste\n\t * des it\u00e9rateurs s\u00e9lectionn\u00e9s.\n\t * <p>\n\t * Plusieurs it\u00e9rateurs peuvent \u00eatre transf\u00e9r\u00e9s si le plus petit \u00e9l\u00e9ment rendu est rendu par plusieurs it\u00e9rateurs\n\t * et que l'indicateur <tt>skipSameElements</tt> est vrai.\n\t *\n\t * @return le plus petit \u00e9l\u00e9ment rendu par les it\u00e9rateurs qui se trouvent dans l'ensemble tri\u00e9.\n\t */\n\tprotected E selectIterators()\n\t{\n\t\t// R\u00e9cup\u00e9ration du premier it\u00e9rateur de l'ensemble tri\u00e9 (celui qui a rendu le plus petit \u00e9l\u00e9ment).\n\t\tIterator<SortableIterator<E>> iteratorsIter = sortedIterators.iterator();\n\t\tSortableIterator<E> currentIter = iteratorsIter.next();\n\t\titeratorsIter.remove();\n\t\tselectedIterators.add( currentIter );\n\t\tE lowerElement = currentIter.nextElement;\n\t\tif ( skipSameElements )\n\t\t{\n\t\t\t// Si on souhaite ne pas it\u00e9rer sur les doublons, on s\u00e9lectionne tous les it\u00e9rateurs rendant le\n\t\t\t// m\u00eame \u00e9l\u00e9ment.\n\t\t\twhile ( iteratorsIter.hasNext() )\n\t\t\t{\n\t\t\t\tcurrentIter = iteratorsIter.next();\n\t\t\t\tif ( cmp.compare( lowerElement, currentIter.nextElement ) == 0 )\n\t\t\t\t{\n\t\t\t\t\titeratorsIter.remove();\n\t\t\t\t\tselectedIterators.add( currentIter );\n\t\t\t\t}\n\t\t\t}\n\t\t}\n\n\t\treturn lowerElement;\n\t}\n\n\n\tprotected List<Iterator<E>> newIteratorsList() { return new ArrayList<Iterator<E>>(); }\n\n\tprotected List<SortableIterator<E>> newSortableIteratorsList() { return new ArrayList<SortableIterator<E>>(); }\n\n\n\tprotected SortedSet<SortableIterator<E>> newIteratorsSortedSet( Comparator<? super E> comparator )\n\t{\n\t\treturn new TreeSet<SortableIterator<E>>( new SortableIteratorComparator<E>( comparator ) );\n\t}\n\n\n\tprotected SortableIterator<E> newSortableIterator( Iterator<? extends E> iter )\n\t{\n\t\treturn new SortableIterator<E>( ++iterCount, iter );\n\t}\n\n\n\t/**\n\t * Impl\u00e9mentation d'un it\u00e9rateur qui peut \u00eatre ins\u00e9r\u00e9 dans un ensemble tri\u00e9.\n\t * <p>\n\t * Cet it\u00e9rateur s'appuie sur l'it\u00e9rateur pass\u00e9 au constructeur afin d'impl\u00e9menter les m\u00e9thodes <tt>hasNext</tt>,\n\t * <tt>next</tt> et <tt>remove</tt> (d\u00e9l\u00e9gation).\n\t * <p>\n\t * En plus, cet it\u00e9rateur poss\u00e8de deux champs (<tt>nextElement</tt> et <tt>rank</tt>), qui lui permettent d'\u00eatre\n\t * tri\u00e9 par un comparateur de type <tt>ElementComparator</tt>.\n\t * <p>\n\t * <tt>nextElement</tt> re\u00e7oit l'\u00e9l\u00e9ment rendu par la m\u00e9thode <tt>next</tt> \u00e0 chaque fois qu'elle est invoqu\u00e9e. Ce\n\t * champ permet d'ordonner les it\u00e9rateurs en pla\u00e7ant en premier celui qui a rendu le plus petit \u00e9l\u00e9ment. \n\t * <p>\n\t * <tt>rank</tt> est un num\u00e9ro d'ordre utilis\u00e9 pour comparer deux <tt>SortableIterator</tt> dont les\n\t * <tt>nextElement</tt> sont \u00e9gaux.\n\t */\n\tprotected static class SortableIterator<E> implements Iterator<E>\n\t{\n\t\t// Iterateur vers lequel les m\u00e9thode hasNext, next et remove doivent \u00eatre d\u00e9l\u00e9gu\u00e9es.\n\t\tprotected Iterator<? extends E> iterator;\n\n\t\t// Element rendu par le dernier appel \u00e0 next.\n\t\tprotected E nextElement;\n\n\t\t// Num\u00e9ro d'ordre affect\u00e9 \u00e0 la construction.\n\t\tprotected int rank;\n\n\n\t\tpublic SortableIterator( int count, Iterator<? extends E> iter )\n\t\t{\n\t\t\tsuper();\n\t\t\titerator = iter;\n\t\t\trank = count;\n\t\t}\n\n\n\t\tpublic boolean hasNext() { return iterator.hasNext(); }\n\n\t\tpublic E next() { return nextElement = iterator.next(); }\n\n\t\tpublic void remove() { iterator.remove(); }\n\t}\n\n\n\t/**\n\t * Comparateur utilis\u00e9 pour comparer des <tt>SortableIterator</tt>.\n\t * <p>\n\t * Ce comparateur s'appuie sur un comparateur d'\u00e9l\u00e9ments fourni au constructeur.\n\t * <p>\n\t * La comparaison consiste \u00e0 compararer d'abord les <tt>nextElement</tt> de chaque <tt>SortableIterator</tt> en\n\t * utilisant le comparateur d'\u00e9l\u00e9ments fourni, puis de comparer ensuite les <tt>rank</tt> de chaque it\u00e9rateur si\n\t * les <tt>nextElement</tt> sont \u00e9gaux.\n\t *\n\t */\n\tprotected static class SortableIteratorComparator<E> implements Comparator<SortableIterator<E>>\n\t{\n\t\tprotected Comparator<? super E> cmp;\n\n\t\tpublic SortableIteratorComparator( Comparator<? super E> comparator )\n\t\t{\n\t\t\tsuper();\n\t\t\tcmp = comparator;\n\t\t}\n\n\t\tpublic int compare( SortableIterator<E> left, SortableIterator<E> right )\n\t\t{\n\t\t\tint result = cmp.compare( left.nextElement, right.nextElement );\n\t\t\treturn result == 0 ? left.rank-right.rank : result;\n\t\t}\n\t}\n}\n";
    }

    public static void main(String[] args) {
        long seed = System.currentTimeMillis();
        random = new RandomBuilder(seed);
        fmt = new Formatter(System.out);
        fmt.format("Start TextAnalyserTest seed=%d\n", seed);
        testCount = 1;
        long time = System.currentTimeMillis();
        try {
            TextPartitionTest.test(seed);
        }
        catch (Throwable t) {
            t.printStackTrace(System.out);
        }
        time = System.currentTimeMillis() - time;
        fmt.format("Stop TextAnalyserTest, %d checkings done in %d ms.\n", checkCount, time);
    }

    protected static void test(long seed) {
        random.setSeed(seed);
        Dictionary dictionary = new Dictionary();
        DefaultTextPartitioner partitioner1 = new DefaultTextPartitioner();
        BasicTextPartition basicPartition = new BasicTextPartition(dictionary, (ITextPartitioner)partitioner1);
        basicPartition.setText(initialText);
        DefaultTextPartitioner partitioner2 = new DefaultTextPartitioner();
        IncrementalTextPartition incrementalPartition = new IncrementalTextPartition(dictionary, (ITextPartitioner)partitioner2);
        incrementalPartition.setText(initialText);
        int wordCount1 = basicPartition.getWordsCount();
        int nbWords = dictionary.nbOfWord();
        int textLength = initialText.length();
        fmt.format("TextLength=%d, DistinctWords=%d, WordsCount=%d\n", textLength, nbWords, wordCount1);
        TextPartitionTest.test(basicPartition, incrementalPartition);
        int i = 0;
        while (i < 10) {
            int beginIdx = random.nextInt(textLength + 1);
            int endIdx = random.nextInt(textLength + 1);
            if (endIdx < beginIdx) {
                int tmp = endIdx;
                endIdx = beginIdx;
                beginIdx = tmp;
            }
            ISubTextPartition subPart1 = basicPartition.subTextPartition(beginIdx, endIdx);
            ISubTextPartition subPart2 = incrementalPartition.subTextPartition(beginIdx, endIdx);
            TextPartitionTest.test(subPart1, subPart2);
            ++i;
        }
    }

    protected static void test(ITextPartition partition1, ITextPartition partition2) {
        TextPartitionTest.check(Strings.sameCharSequences((CharSequence)partition1.getText(), (CharSequence)partition2.getText()), "Not same text", new Object[0]);
        int wordCount1 = partition1.getWordsCount();
        int wordCount2 = partition2.getWordsCount();
        TextPartitionTest.check(wordCount1 == wordCount2, "Not same word count", new Object[0]);
        TextPartitionTest.checkWords(partition1, partition2);
        IntSequence initialWords = partition1.getWords();
        TextPartitionTest.checkModify1(initialWords, partition1, partition2);
        TextPartitionTest.checkModify2(initialWords, partition1, partition2);
    }

    protected static void checkWords(ITextPartition partition1, ITextPartition partition2) {
        int wordCount2;
        int wordCount1 = partition1.getWordsCount();
        TextPartitionTest.check(wordCount1 == (wordCount2 = partition2.getWordsCount()), "Not same word count", new Object[0]);
        IntSequence words1 = partition1.getWords();
        IntSequence words2 = partition2.getWords();
        int wordRank = 0;
        while (wordRank < wordCount1) {
            int wordId = words1.intAt(wordRank);
            TextPartitionTest.check(wordId == words2.intAt(wordRank), "Wrong word", new Object[0]);
            int wordLength = partition1.wordLength(wordRank);
            TextPartitionTest.check(partition2.wordLength(wordRank) == wordLength, "Wrong word length", new Object[0]);
            int wordIdx = partition1.wordBeginIndex(wordRank);
            TextPartitionTest.check(partition2.wordBeginIndex(wordRank) == wordIdx, "Wrong word index", new Object[0]);
            int spaceBefore = partition1.spaceBeforeWord(wordRank);
            TextPartitionTest.check(partition2.spaceBeforeWord(wordRank) == spaceBefore, "Wrong space before word", new Object[0]);
            int spaceAfter = partition1.spaceAfterWord(wordRank);
            TextPartitionTest.check(partition2.spaceAfterWord(wordRank) == spaceAfter, "Wrong space after word", new Object[0]);
            int rank1 = partition1.wordRankFromIndex(wordIdx);
            TextPartitionTest.check(rank1 == wordRank, "Wrong rank from index, analyser1", new Object[0]);
            int rank2 = partition2.wordRankFromIndex(wordIdx);
            TextPartitionTest.check(rank2 == wordRank, "Wrong rank from index, analyser2", new Object[0]);
            if (wordIdx > 0) {
                rank1 = partition1.wordRankFromIndex(wordIdx - 1);
                TextPartitionTest.check(rank1 < 0 ? ~rank1 == wordRank : rank1 == wordRank - 1, "Wrong rank from previous index, analyser1", new Object[0]);
                rank2 = partition2.wordRankFromIndex(wordIdx - 1);
                TextPartitionTest.check(rank2 < 0 ? ~rank2 == wordRank : rank2 == wordRank - 1, "Wrong rank from previous index, analyser2", new Object[0]);
            }
            if (wordLength > 1) {
                rank1 = partition1.wordRankFromIndex(wordIdx + 1);
                TextPartitionTest.check(rank1 == wordRank, "Wrong rank from next index, analyser1", new Object[0]);
                rank2 = partition2.wordRankFromIndex(wordIdx + 1);
                TextPartitionTest.check(rank2 == wordRank, "Wrong rank from next index, analyser2", new Object[0]);
            }
            TextPartitionTest.check((rank1 = partition1.wordRankFromIndex(wordIdx + wordLength)) < 0 ? ~rank1 == wordRank + 1 : rank1 == wordRank + 1, "Wrong rank from last index, analyser1", new Object[0]);
            rank2 = partition2.wordRankFromIndex(wordIdx + wordLength);
            TextPartitionTest.check(rank2 < 0 ? ~rank2 == wordRank + 1 : rank2 == wordRank + 1, "Wrong rank from last index, analyser2", new Object[0]);
            ++wordRank;
        }
    }

    protected static void checkModify1(IntSequence initialWords, ITextPartition partition1, ITextPartition partition2) {
        int textLength = partition1.getTextLength();
        int modifCount = 0;
        while (modifCount < 10) {
            ++testCount;
            int modLength = random.nextInt(1 + textLength);
            int modIdx = random.nextInt(1 + textLength - modLength);
            CharSequence oldText = partition1.getTextInterval(modIdx, modIdx + modLength);
            partition1.replaceText(modIdx, modIdx + modLength, "");
            partition2.replaceText(modIdx, modIdx + modLength, "");
            TextPartitionTest.checkWords(partition1, partition2);
            partition1.replaceText(modIdx, modIdx, oldText);
            partition2.replaceText(modIdx, modIdx, oldText);
            TextPartitionTest.checkWords(partition1, partition2);
            TextPartitionTest.check(Ints.sameIntSequences((IntSequence)partition1.getWords(), (IntSequence)initialWords), "Wrong word sequence", new Object[0]);
            ++modifCount;
        }
    }

    protected static void checkModify2(IntSequence initialWords, ITextPartition partition1, ITextPartition partition2) {
        int textLength = partition1.getTextLength();
        int modifCount = 0;
        while (modifCount < 10) {
            ++testCount;
            int modLength = random.nextInt(1 + textLength);
            int modIdx = random.nextInt(1 + textLength - modLength);
            CharSequence oldText = partition1.getTextInterval(modIdx, modIdx + modLength);
            int copyLength = random.nextInt(1 + textLength);
            int copyIdx = random.nextInt(1 + textLength - copyLength);
            CharSequence newText = partition1.getTextInterval(copyIdx, copyIdx + copyLength);
            partition1.replaceText(modIdx, modIdx + modLength, newText);
            partition2.replaceText(modIdx, modIdx + modLength, newText);
            TextPartitionTest.checkWords(partition1, partition2);
            partition1.replaceText(modIdx, modIdx + copyLength, oldText);
            partition2.replaceText(modIdx, modIdx + copyLength, oldText);
            TextPartitionTest.checkWords(partition1, partition2);
            TextPartitionTest.check(Ints.sameIntSequences((IntSequence)partition1.getWords(), (IntSequence)initialWords), "Wrong word sequence", new Object[0]);
            ++modifCount;
        }
    }

    protected static int perfModify(ITextPartition partition) {
        int checkSum = 0;
        int size = 0;
        while (size < 7) {
            int textLength = partition.getTextLength();
            int maxModifLength = Math.min(textLength, 10);
            int modifCount = 0;
            while (modifCount < 100) {
                ++testCount;
                int modLength = random.nextInt(maxModifLength);
                int modIdx = random.nextInt(1 + textLength - modLength);
                CharSequence oldText = partition.getTextInterval(modIdx, modIdx + modLength);
                int copyLength = random.nextInt(maxModifLength);
                int copyIdx = random.nextInt(1 + textLength - copyLength);
                CharSequence newText = partition.getTextInterval(copyIdx, copyIdx + copyLength);
                int i = 0;
                while (i < 10) {
                    partition.replaceText(modIdx, modIdx + modLength, newText);
                    partition.replaceText(modIdx, modIdx + copyLength, oldText);
                    ++i;
                }
                int wordCount = partition.getWordsCount();
                int wordRank = random.nextInt(wordCount);
                int wordId = partition.wordIdAt(wordRank);
                int wordIdx = partition.wordBeginIndex(wordRank);
                int spaceBefore = partition.spaceBeforeWord(wordRank);
                int spaceAfter = partition.spaceAfterWord(wordRank);
                wordRank = partition.wordRankFromIndex(wordIdx);
                checkSum += wordCount + wordId + wordIdx + spaceBefore + spaceAfter + wordRank;
                ++modifCount;
            }
            partition.replaceText(textLength, textLength, partition.getText());
            ++size;
        }
        return checkSum;
    }

    protected static void check(boolean condition, String message, Object ... args) {
        ++checkCount;
        if (!condition) {
            StringBuilder builder = new StringBuilder(message.length() + 32);
            Formatter strFmt = new Formatter(builder);
            strFmt.format("Wrong check %d : ", checkCount);
            strFmt.format(message, args);
            throw new RuntimeException(builder.toString());
        }
    }
}

